#!/bin/bash

# Copyright 2009-2019 Broadcom. All Rights Reserved.
# The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.

KITNAME=elxflashOffline
SUCCESS=0
FAILURE=1
FALSE=0
TRUE=1

KIT_INSTALL_STATUS=$SUCCESS

CLU_OS=""
CLU_ARCH=""
CLU_OS_FULL="NONE"


LOGFILE_NAME="offline-install.log"
LOGFILE="/usr/sbin/linlpcfg/${LOGFILE_NAME}"
# Function:  log_message()
#
# Description:
#   Print the strings passed as parameters to both STDOUT as well as
#   to a log file.
#
# Parameters:
#   A series of strings.
#
# Returns:
#   0 on success.
log_message()
{
    local ECHO_OPTIONS
    LOGDIRECTORY=${LOGFILE%/*}
    if [ ! -d ${LOGDIRECTORY} ] ; then
        mkdir -p ${LOGDIRECTORY}
    fi

    if [ $# -eq 0 ] ; then
        echo "" | tee -a ${LOGFILE}
    elif [ -z "$1" ] ; then
        echo "" | tee -a ${LOGFILE}
    elif [ "$1" = "-init" ] ; then
        echo "$0 - $(date)" >> ${LOGFILE}
    else
        while [ -n "$1" ] ; do
            if [ "${1:0:1}" = "-" ] ; then
                ECHO_OPTIONS="${1}"
            else
                echo ${ECHO_OPTIONS} "$1" | tee -a ${LOGFILE}
            fi
            shift
        done
    fi
    return $SUCCESS
}


#
# Print a OneConnect software message
#
ccx_adapter_msg()
{
    log_message "If you also have OneConnect adapters installed, please download and install the latest OneConnect software kits."
}


#
# Determine if this is a supported operating system
#
determine_os()
{
    if [ -f /etc/redhat-release ]; then
        # check for oracle
        oracleos="FALSE"
        if [ -f /etc/issue ]; then
            oracleos=`grep Oracle /etc/issue | wc -l | awk '{print $1 == "1" ? "TRUE" : "FALSE"}'`
        fi
        if [ "$oracleos" == "FALSE" ]; then
            if [ -f /etc/oracle-release ]; then
                oracleos=`grep Oracle /etc/oracle-release | wc -l | awk '{print $1 == "1" ? "TRUE" : "FALSE"}'`
            fi
        fi
        
        citrixos="FALSE"
        grep -q Citrix /etc/issue || grep -q XenServer /etc/issue
        if [ $? -eq 0 ]; then
            citrixos="TRUE"
        fi
        
        centos=`grep CentOS /etc/redhat-release | wc -l`
        scientificos=`grep Scientific /etc/redhat-release | wc -l`
        
        if [ "$centos" = "1" ]; then
            RHEL_OS_STR="CentOS"
        elif [ "$scientificos" = "1" ]; then
            RHEL_OS_STR="Scientific Linux"
        else
            RHEL_OS_STR="Red Hat Enterprise Linux"
        fi
        
        if [ "$citrixos" = "TRUE" ]; then
            log_message "Citrix XenServer is not supported."
            log_message "Aborting installation."
            exit $FAILURE
        
        elif [ "$oracleos" = "TRUE" ]; then
            kernel_major_version=$(uname -r | awk -F- '{print$1}' | awk -F. '{print$3}')
            kernel_minor_version=$(uname -r | awk -F- '{print$2}' | awk -F. '{print$1}')
            
            uekos=`uname -a | grep uek | wc -l`
            
            if [ "$uekos" = "1" ]; then
                oracle_major_version=`cat /etc/oracle-release | cut -d' ' -f5 | cut -d. -f1`
                oracle_minor_version=`cat /etc/oracle-release | cut -d' ' -f5 | cut -d. -f2`
                oracle_vm_server=`cat /etc/oracle-release | grep VM | wc -l`
                
                if [ "$oracle_major_version" -eq 7 ] && [ "$oracle_minor_version" -ge 5 ]; then
                    CLU_OS="uek-7"
                    CLU_OS_FULL="uek-7.${oracle_minor_version}"
                else
                    log_message "This is not a supported version of Oracle Enterprise Linux"
                    log_message "Aborting installation."
                    exit $FAILURE
                fi
            # R1 RH5.6 and R1 RH6.0 uek installations are not detected in the logic above
            # check for these two now
            elif [ "$kernel_major_version" -eq 32 ] && [ "$kernel_minor_version" -eq 100 ]; then
                oracle_major_version=`cat /etc/redhat-release | cut -d' ' -f7 | cut -d. -f1`
                oracle_minor_version=`cat /etc/redhat-release | cut -d' ' -f7 | cut -d. -f2`
                
                if [ "$oracle_major_version" -eq 7 ] && [ "$oracle_minor_version" -ge 5 ]; then
                    CLU_OS="uek-7"
                    CLU_OS_FULL="uek-7.${oracle_minor_version}"
                else
                    log_message "This is not a supported version of Oracle Enterprise Linux"
                    log_message "Aborting installation."
                    exit $FAILURE
                fi
            else
                oracle_major_version=`cat /etc/redhat-release | awk -F'.' '{print $1}' | awk '{print $NF}'`
                oracle_minor_version=`cat /etc/redhat-release | awk -F'.' '{print $2}' | cut -d' ' -f1`
                
                # else these are oracle OEL installations and not UEK; therefore install rhel binaries
                if [ "$oracle_major_version" -eq 8 ] && [ "$oracle_minor_version" -ge 0 ]; then
                    CLU_OS="rhel-8"
                    CLU_OS_FULL="rhel-8.${oracle_minor_version}"
                elif [ "$oracle_major_version" -eq 7 ] && [ "$oracle_minor_version" -ge 5 ]; then
                    CLU_OS="rhel-7"
                    CLU_OS_FULL="rhel-7.${oracle_minor_version}"
                else
                    log_message "This is not a supported version of Oracle Enterprise Linux"
                    log_message "Aborting installation."
                    exit $FAILURE
                fi
            fi

        else
            rhel_major_version=`cat /etc/redhat-release | awk -F'.' '{print $1}' | awk '{print $NF}'`
            rhel_minor_version=`cat /etc/redhat-release | awk -F'.' '{print $2}' | cut -d' ' -f1`
            
            if [ "$rhel_major_version" -eq 7 ] && [ "$rhel_minor_version" -ge 5 ]; then
                CLU_OS="rhel-7"
                CLU_OS_FULL="rhel-7.${rhel_minor_version}"
            elif [ "$rhel_major_version" -eq 8 ] && [ "$rhel_minor_version" -ge 0 ]; then
                CLU_OS="rhel-8"
                CLU_OS_FULL="rhel-8.${rhel_minor_version}"
            else
                log_message "This is not a supported version of $RHEL_OS_STR."
                log_message "Aborting installation."
                exit $FAILURE
            fi
        fi
    elif [ -f /etc/SuSE-release ]; then
        sles_major_version=`cat /etc/SuSE-release | grep VERSION | awk '{ print $3 }'`
        sles_minor_version=`cat /etc/SuSE-release | grep PATCHLEVEL | awk '{ print $3 }'`
        if [ "$sles_major_version" -eq 12 ] && [ "$sles_minor_version" -ge 5 ]; then
            CLU_OS="sles-12-sp5"
            CLU_OS_FULL="sles-12.${sles_minor_version}"
        elif [ "$sles_major_version" -eq 12 ] && [ "$sles_minor_version" -eq 4 ]; then
            CLU_OS="sles-12-sp4"
            CLU_OS_FULL="sles-12.${sles_minor_version}"
        else
            log_message "This is not a supported version of SuSE Linux Enterprise Server."
            log_message "Aborting installation."
            exit $FAILURE
        fi
    elif [ -f /etc/os-release ]; then
        version_file="/etc/os-release"
        sles_str=`cat $version_file | grep ^NAME= | cut -f2 -d'"'`
        if [ -z $sles_str ] || [ "$sles_str" != "SLES" ]; then
            log_message "Cannot determine Linux distribution."
            log_message "Aborting installation."
            exit $FAILURE
        fi
        
        sles_version=`cat $version_file | grep ^VERSION_ID= | cut -f2 -d'"'`
        sles_major_version=`cat $version_file | grep ^VERSION_ID= | cut -f2 -d'"' | cut -f1 -d "."`
        if [[ $sles_version == *"."* ]]; then
            sles_minor_version=`cat $version_file | grep ^VERSION_ID= | cut -f2 -d'"' | cut -f2 -d "."`
        else
            sles_minor_version=0
        fi
        
        if [ "$sles_major_version" -eq 12 ] && [ "$sles_minor_version" -ge 5 ]; then
            CLU_OS="sles-12-sp5"
            CLU_OS_FULL="sles-12.${sles_minor_version}"
        elif [ "$sles_major_version" -eq 12 ] && [ "$sles_minor_version" -eq 4 ]; then
            CLU_OS="sles-12-sp4"
            CLU_OS_FULL="sles-12.${sles_minor_version}"
        elif [ "$sles_major_version" -eq 15 ]; then
            CLU_OS="sles-15-sp0"
            CLU_OS_FULL="sles-15.${sles_minor_version}"
        else
            log_message "This is not a supported version of SuSE Linux Enterprise Server."
            log_message "Aborting installation."
            exit $FAILURE
        fi
    else
        log_message "Cannot determine Linux distribution."
        log_message "Aborting installation."
        exit $FAILURE
    fi
    
    if [ ! -d ${CLU_ARCH}/${CLU_OS} ]; then
        log_message "Could not find installation directory: $CLU_OS/$CLU_ARCH"
        log_message "Aborting installation."
        exit $FAILURE
    fi
}


#
# Determine processor architecture
#
determine_arch()
{
    CLU_ARCH=$(uname -m)
    case ${CLU_ARCH} in
        i[3456]86)
            CLU_ARCH=i386
            ;;
        
        x86_64)
            CLU_ARCH=x86_64
            ;;
        
        ppc64)
            CLU_ARCH=ppc64
            ;;
            
        *)
            log_message "Unsupported architecture: $CLU_ARCH"
            log_message "Aborting installation."
            exit $FAILURE
            ;;
    esac
}


#
# Exit if a HBAnyware Core or Enterprise Kit is detected
#
function verify_no_hbanyware()
{
    rpm -q --quiet HBAnyware
    if [ $? -eq 0  ]; then
        HBANYWARE_ENT_RPM=`rpm -q HBAnyware`
        log_message "HBAnyware Enterprise kit: $HBANYWARE_ENT_RPM detected!"
        log_message "Please uninstall the HBAnyware Enterprise Kit before running this script."
        log_message "Aborting installation."
        exit $FAILURE
    fi
    rpm -q --quiet elxlinuxcorekit
    if [ $? -eq 0 ]; then
        HBANYWARE_CORE_RPM=`rpm -q elxlinuxcorekit`
        log_message "HBAnyware Core kit: $HBANYWARE_CORE_RPM detected!"
        log_message "Please uninstall the HBAnyware Core Kit before running this script."
        log_message "Aborting installation."
        exit $FAILURE
    fi
    return $SUCCESS
}


#
# Verify that no dependencies or conflicts exist
#
function verify_dependencies()
{
    if [ "$KITNAME" = "elxlinlpcfg" ]; then
        rpm -U --test --replacepkgs ${CLU_ARCH}/${CLU_OS}/elxlinlpcfg-*.rpm
        if [ $? -ne 0 ]; then
            exit $FAILURE
        fi
    elif [ "$KITNAME" = "elxflashOffline" ]; then
        rpm -U --test --replacepkgs ${CLU_ARCH}/${CLU_OS}/elxlinlpcfg-*.rpm ${CLU_ARCH}/${CLU_OS}/elxflashOffline-*.rpm
        if [ $? -ne 0 ]; then
            exit $FAILURE
        fi
    fi
}


#
# Install install_elxflashOffline
#
function install_elxflashOffline()
{
    ELXLPCFGRPM=""
    ELXFLASHOFFLINERPM=""
    
    if [ -f elxlinlpcfg-*.rpm ]; then
        ELXLPCFGRPM=`ls elxlinlpcfg-*.rpm`
    fi
    if [ -f elxflashOffline-*.rpm ]; then
        ELXFLASHOFFLINERPM=`ls elxflashOffline-*.rpm`
    fi
    
    if [ -n "$ELXLPCFGRPM" ] && [ -n "$ELXFLASHOFFLINERPM" ]; then
        log_message "Installing elxlinlpcfg and elxflashOffline..."
        log_message "Installing ./${CLU_ARCH}/${CLU_OS}/${ELXLPCFGRPM}..."
        log_message "Installing ./${CLU_ARCH}/${CLU_OS}/${ELXFLASHOFFLINERPM}..."
        rpm -U --replacepkgs elxlinlpcfg-*.rpm elxflashOffline-*.rpm
        STATUS=$?
        if [ $STATUS -ne 0 ]; then
            log_message "elxflashOffline install failed."
            KIT_INSTALL_STATUS=$FAILURE
        fi
    else
        log_message "elxflashOffline install failed."
        KIT_INSTALL_STATUS=$FAILURE
    fi
}


#
# The installation is complete, check the status of each installation
#
function prepare_status_message()
{
    if [ $KIT_INSTALL_STATUS -ne $SUCCESS ]; then
        log_message ""
        log_message "$KITNAME install failed."
        log_message ""
        exit $FAILURE
    else
        log_message ""
        log_message "$KITNAME install completed successfully."
        ccx_adapter_msg
        log_message ""
        exit $SUCCESS
    fi
}


#
# Begin script execution
#
determine_arch
determine_os


#
# Begin the installation: CD to the install directory
#
DIR=$(cd ${0%/*} ; pwd )
cd $DIR


#
# Initialize the log file.
#
log_message -init


#
# Perform additional installation checks
#
if [ "$KITNAME" != "elxflashOffline_NIC_Only" ]; then
    verify_no_hbanyware
fi


#
# Perform additional installation checks
#
verify_dependencies


#
# Install the drivers
#
log_message "Beginning $KITNAME installation..."


#
# Install the applications
#
cd ${CLU_ARCH}/${CLU_OS}
install_elxflashOffline
cd ../../


#
# The installation is complete, check the status of each installation
#
prepare_status_message

